//
//  TodoList.swift
//  Do It
//
//  Created by Jim Dovey on 8/25/19.
//  Copyright © 2019 Jim Dovey. All rights reserved.
//

import SwiftUI
import Combine

struct ErrorInfo: Identifiable {
    let error: Error
    let id: UUID = UUID()
    
    init(_ error: Error) {
        self.error = error
    }
}

struct TodoList: View {
    private enum ListData {
        case list(TodoItemList)
        case group(TodoItemGroup)
    }
    @State private var sortBy: SortOption = .manual
    @State private var showingChooser: Bool = false
    @Environment(\.presentationMode) private var presentationMode
    @EnvironmentObject private var data: DataCenter

    private static let itemTemplate = TodoItem(
        title: "New Item", priority: .normal, listID: UUID())
    
    @State private var editingItem = Self.itemTemplate
    @State private var listData: ListData
    
    private enum EditorID: Identifiable, Hashable {
        case itemEditor
        case listEditor
        
        var id: EditorID { self }
    }
    
    @State private var presentedEditor: EditorID? = nil
    
    @Environment(\.errorPublisher) private var errorPublisher
    
    @State private var selectedItems: Set<UUID> = []
    @Environment(\.undoManager) var undoManager

    init(list: TodoItemList) {
        self._listData = State(wrappedValue: .list(list))
    }

    init(group: TodoItemGroup) {
        self._listData = State(wrappedValue: .group(group))
    }

    var body: some View {
        List(selection: $selectedItems) {
            ForEach(sortedItems) { item in
                NavigationLink(
                    destination: TodoItemDetail(item: item)
                        .environmentObject(self.data)
                ) {
                    TodoItemRow(item: item)
                        .accentColor(self.color(for: item))
                }
            }
            .onDelete { self.removeTodoItems(atOffsets: $0) }
            .onMove(perform: self.sortBy == .manual && isList
                ? { self.moveTodoItems(fromOffsets: $0, to: $1) }
                : nil)
        }
        .navigationBarTitle(title)
        .navigationBarItems(trailing: barItems)
        .listStyle(GroupedListStyle())
        .actionSheet(isPresented: $showingChooser) {
            ActionSheet(
                title: Text("Sort Order"),
                buttons: SortOption.allCases.map { opt in
                    ActionSheet.Button.default(Text(opt.title)) {
                        self.sortBy = opt
                    }
            })
        }
        .sheet(item: $presentedEditor) { which -> AnyView in
            switch which {
            case .itemEditor:
                return AnyView(
                    self.editorSheet
                        .environmentObject(self.data)
                )
            case .listEditor:
                return AnyView(
                    TodoListEditor(list: self.list!)
                        .environmentObject(self.data)
                )
            }
        }
        .onReceive(data.$todoItems.combineLatest(data.$todoLists).receive(on: RunLoop.main)) { _ in
            self.updateData()
        }
    }
}

// MARK: - Helper Properties

extension TodoList {
    private var sortButton: some View {
        Button(action: { self.showingChooser.toggle() }) {
            Image(systemName: "arrow.up.arrow.down.circle.fill")
                .imageScale(.large)
                .accessibility(label: Text("Sort List"))
        }
    }

    private var addButton: some View {
        Button(action: {
            self.editingItem = Self.itemTemplate
            self.editingItem.listID = self.list?.id ?? self.data.defaultListID
            self.presentedEditor = .itemEditor
        }) {
            Image(systemName: "plus.circle.fill")
                .imageScale(.large)
                .accessibility(label: Text("Add New To-Do Item"))
        }
        .accessibility(label: Text("Add a new To-Do Item"))
    }

    private var editorSheet: some View {
        let done = Button(action:{
            self.data.addTodoItem(self.editingItem)
            self.presentedEditor = nil
        }) {
            Text("Done")
                .bold()
        }
        let cancel = Button("Cancel") {
            self.presentedEditor = nil
        }
        return NavigationView {
            TodoItemEditor(item: $editingItem)
                .navigationBarItems(leading: cancel, trailing: done)
        }
    }

    private var barItems: some View {
        HStack(spacing: 14) {
            if isList {
                Button(action: { self.presentedEditor = .listEditor }) {
                    Image(systemName: "info.circle")
                        .imageScale(.large)
                }
            }
            sortButton
            addButton
            EditButton()
        }
    }

    private var isList: Bool {
        if case .list = self.listData {
            return true
        }
        return false
    }
    
    private var list: TodoItemList? {
        if case let .list(list) = self.listData {
            return list
        }
        return nil
    }
    
    private var isAll: Bool {
        if case let .group(g) = self.listData, g == .all {
            return true
        }
        return false
    }

    private func forciblyDismiss() {
        presentationMode.wrappedValue.dismiss()
    }

    private var items: [TodoItem] {
        switch listData {
        case .list(let list): return data.items(in: list)
        case .group(let group): return group.items(from: data)
        }
    }

    private var title: LocalizedStringKey {
        switch listData {
        case .list(let list): return LocalizedStringKey(list.name)
        case .group(let group): return group.title
        }
    }

    private func color(for item: TodoItem) -> Color {
        switch listData {
        case .list(let list): return list.color.uiColor
        case .group(let group): return group.color
        }
    }
}

// MARK: - Sorting

extension TodoList {
    private var sortedItems: [TodoItem] {
        if case .manual = sortBy { return items }

        return items.sorted {
            switch sortBy {
            case .title:
                return $0.title.lowercased() < $1.title.lowercased()
            case .priority:
                return $0.priority > $1.priority
            case .dueDate:
                return ($0.date ?? .distantFuture) <
                    ($1.date ?? .distantFuture)
            case .manual:
                fatalError("unreachable")
            }
        }
    }
}

// MARK: - Model Manipulation

extension TodoList {
    private var usingUnchangedList: Bool {
        sortBy == .manual
    }

    private func removeTodoItems(atOffsets offsets: IndexSet) {
        if let list = list {
            data.removeTodoItems(atOffsets: offsets, in: list)
        }
        else {
            let items = sortedItems
            let uuids = offsets.map { items[$0].id }
            data.removeTodoItems(withIDs: uuids)
        }
    }

    private func moveTodoItems(fromOffsets offsets: IndexSet, to newIndex: Int) {
        guard let list = self.list else { return }
        data.moveTodoItems(fromOffsets: offsets, to: newIndex, within: list)
    }

    private func updateData() {
        switch listData {
        case let .list(list):
            if let newList = data.todoLists.first(where: { $0.id == list.id }) {
                listData = .list(newList)
            }
            else {
                // List is gone!
                forciblyDismiss()
            }

        case .group:
            break
        }
    }
}

fileprivate enum SortOption: String, CaseIterable {
    case title = "Title"
    case priority = "Priority"
    case dueDate = "Due Date"
    case manual = "Manual"

    var title: LocalizedStringKey { LocalizedStringKey(rawValue) }
}

struct TodoList_Previews: PreviewProvider {
    static var previews: some View {
        return NavigationView {
            TodoList(group: .all)
                .environmentObject(DataCenter())
        }
        .previewDevice("iPhone 11 Pro")
    }
}
